{
  config,
  pkgs,
  ...
}: let
  lib = pkgs.lib;
  host = "10.0.${toString vm-index}.1";

  ports = {
    sonarr.port = 8989;
    radarr.port = 7878;
    jackett.port = 9117;
  };

  vm-index = 1;
  vm-mac = "02:00:00:00:00:02";
  vm-name = "auto-torrent";
  vpn-endpoint = "193.32.248.70";
  enable-services = true;
  wg-config = "/mnt/wg.conf";
  jellyfin-gid = 989;

  ports-list = (pkgs.lib.attrsets.mapAttrsToList (name: value: value.port) ports) ++ [9091];
in {
  microvm.autostart = [vm-name];
  imports = [
    ./nginx.nix
    ../modules/microvm.nix
  ];

  users.extraUsers.microvm.extraGroups = [
    "jellyfin" # access to media folder
  ];

  system.activationScripts."make-${vm-name}-data-dir" = lib.stringAfter ["var"] ''
    mkdir -p /var/lib/${vm-name}
    # chown -R microvm:jellyfin /var/lib/${vm-name}
  '';

  microvm.vms.${vm-name} = {
    config = {...}: {
      system.stateVersion = "25.05";
      microvm.interfaces = [
        {
          id = "vm${toString vm-index}";
          type = "tap";
          mac = vm-mac;
        }
      ];

      time.timeZone = "Europe/Copenhagen";

      # 1gb of memory
      microvm.mem = 1024;

      microvm.shares = let
        proto = "virtiofs";
      in [
        {
          tag = "ro-store";
          source = "/nix/store";
          mountPoint = "/nix/.ro-store";
        }
        {
          tag = "data-dir";
          source = "/var/lib/${vm-name}";
          mountPoint = "/mnt";
          inherit proto;
        }
        {
          tag = "media-dir";
          source = "/var/lib/media";
          mountPoint = "/media";
          inherit proto;
        }
      ];

      networking.useNetworkd = true;
      networking.usePredictableInterfaceNames = false;
      systemd.network.networks."10-eth" = {
        matchConfig.MACAddress = vm-mac;
        address = [
          "10.0.${toString vm-index}.1/24"
        ];
        routes = [
          # Default route
          {
            Destination = "${toString vpn-endpoint}/32";
            Gateway = "10.0.${toString vm-index}.254";
            GatewayOnLink = true;
          }
        ];
        networkConfig = {
          DNS = [
            "9.9.9.9"
            "8.8.8.8"
            "8.8.4.4"
          ];
        };
      };

      networking.useDHCP = false;
      networking.nameservers = [
        "10.0.101.1"
        "8.8.8.8"
        "8.8.4.4"
      ];

      systemd.services."start-wireguard" = {
        description = "Start wireguard mullvad";
        after = ["network-online.target"];
        wants = ["network-online.target"];
        wantedBy = ["multi-user.target"];

        serviceConfig = {
          type = "oneshot";
          ExecStart = pkgs.writeShellScript "wgconf.sh" ''
            ${pkgs.wireguard-tools}/bin/wg-quick up ${wg-config}
          '';
          RemainAfterExit = "yes";
        };
      };

      networking.wireguard.enable = true;

      # fuck nano
      programs.nano.enable = lib.mkForce false;
      programs.vim.enable = true;

      # Fix permissions with groups
      users.users = {
        sonarr.extraGroups = ["jellyfin"];
        radarr.extraGroups = ["jellyfin"];
        transmission.extraGroups = ["jellyfin"];
        jackett.extraGroups = ["jellyfin"];
      };

      users.groups.jellyfin = {
        gid = jellyfin-gid;
      };

      # Services
      services.sonarr = pkgs.lib.mkIf enable-services {
        enable = true;
        openFirewall = true;
        dataDir = "/mnt/sonarr";
      };
      services.radarr = pkgs.lib.mkIf enable-services {
        enable = true;
        openFirewall = true;
        dataDir = "/mnt/radarr";
      };
      services.jackett = pkgs.lib.mkIf enable-services {
        enable = true;
        dataDir = "/mnt/jackett";
        openFirewall = true;
      };

      services.transmission = pkgs.lib.mkIf enable-services {
        enable = true;
        openPeerPorts = true;
        openRPCPort = true;
        home = "/mnt/transmission";
        webHome = "${pkgs.flood-for-transmission}";
        settings = {
          download-dir = "/media/.transmission/";
          incomplete-dir = "/media/.transmission/.incomplete";
          peer-port-random-low = 65500;
          peer-port-random-high = 65535;
          peer-port-random-on-start = true;
          download-queue-enabled = false;

          rpc-authentication-required = false;
          rpc-bind-address = "0.0.0.0";
          rpc-host-whitelist = builtins.concatStringsSep "," [host "127.0.0.1" "10.0.101.10" "10.0.${toString vm-index}.254"];
          rpc-whitelist = builtins.concatStringsSep "," [host "127.0.0.1" "10.0.101.10" "10.0.${toString vm-index}.254"];
        };
        downloadDirPermissions = "775";
        performanceNetParameters = true;
      };

      # debugging
      users.users.root = {
        openssh.authorizedKeys.keys = [
          "ssh-ed25519 AAAAC3NzaC1lZDI1NTE5AAAAIC562Woe/yT/3dNVceN9rKPJQcvgTFzIhJVdVGv7sqn1 baritone@server"
        ];
      };

      environment.systemPackages = with pkgs; [
        wireguard-tools
        tcpdump
      ];

      services.openssh = {
        enable = true;
        settings = {
          PermitRootLogin = "yes";
          AllowUsers = null;
          PasswordAuthentication = true;
          KbdInteractiveAuthentication = true;
        };
      };
    };
  };

  # Add virtual hosts for all the different services running in the vm
  services.nginx.virtualHosts =
    builtins.listToAttrs (pkgs.lib.attrsets.mapAttrsToList (
        name: value: {
          name = "${name}.iouhase.net";
          value = {
            forceSSL = true;
            enableACME = true;
            locations."/" = {
              proxyPass = "http://${host}:${toString value.port}";
              # https://github.com/Sonarr/Sonarr/issues/4329
              extraConfig = ''
                send_timeout 100m;
                proxy_connect_timeout 600;
                proxy_send_timeout 600;
                proxy_read_timeout 30m;
              '';
            };
          };
        }
      )
      ports)
    // {
      # Forward transmission web port to vm
      "transmission" = {
        listen = [
          {
            port = 9091;
            addr = "0.0.0.0";
            ssl = false;
          }
        ];
        locations."/" = {
          extraConfig = ''
            proxy_set_header Host $host;
            proxy_set_header X-Real-Ip $remote_addr;
            client_max_body_size 100M;
            proxy_pass http://10.0.${toString vm-index}.1:9091;
          '';
        };
      };
    };

  networking.firewall.allowedTCPPorts = ports-list ++ [9091];
  networking.firewall.allowedUDPPorts = ports-list ++ [9091];
}
