{
  pkgs,
  lib,
  config,
  ...
}: let
  jails = {
    forgejo = mkJail "forgejo" "^.*Failed authentication attempt for .* from <ADDR>.*$";
    jellyfin = mkJail "jellyfin" "^.*Authentication request for .* has been denied \\(IP: <ADDR>\\)\\.";
    sonarr = mkJail "sonarr" "^.*Auth-Failure ip <ADDR> username.*$";
    radarr = mkJail "sonarr" "^.*Auth-Failure ip <ADDR> username.*$";
    vaultwarden = mkJail "vaultwarden" "^.*Username or password is incorrect. Try again. IP: <ADDR>\\. Username: .*$";
    jellyseerr = mkJail "jellyseerr" "^.*Failed login attempt.*{\"ip\":\"<ADDR>\".*$";
    grafana = mkJail "grafana" "^.*path=/login.*remote_addr=<ADDR>.*errorReason=Unauthorized.*errorMessageID=password-auth.failed.*invalid password.*$";
  };

  mkJail = name: filter: {
    jail.settings = {
      enabled = true;
      filter = name;
      backend = "systemd";
      port = "80,443";
      maxretry = 8;
      bantime = "24h";
      findtime = "30m";
      journalmatch = "_SYSTEMD_UNIT=${name}.service";
    };
    filter = ''
      [Definition]
      failregex = ${filter}
    '';
  };
in {
  services.fail2ban = {
    enable = true;

    bantime = "24h";
    bantime-increment = {
      enable = true;
      formula = "ban.Time * math.exp(float(ban.Count+1)*banFactor)/math.exp(banFactor)";
      maxtime = "6969h";
      overalljails = true;
    };

    jails =
      {
        dovecot = lib.mkIf config.services.dovecot2.enable {
          settings = {
            # block IPs which failed to log-in
            # aggressive mode add blocking for aborted connections
            filter = "dovecot[mode=aggressive]";
            maxretry = 3;
          };
        };
      }
      // (lib.attrsets.mapAttrs (name: value: value.jail) jails);
  };

  environment.etc =
    {
      # Defines a filter that detects URL probing by reading the Nginx access log
      "fail2ban/filter.d/nginx-url-probe.local".text = lib.mkDefault (lib.mkAfter ''
        [Definition]
        failregex = ^<HOST>.*(GET /(wp-|admin|boaform|phpmyadmin|\.env|\.git)|\.(dll|so|cfm|asp)|(\?|&)(=PHPB8B5F2A0-3C92-11d3-A3A9-4C7B08C10000|=PHPE9568F36-D428-11d2-A769-00AA001ACF42|=PHPE9568F35-D428-11d2-A769-00AA001ACF42|=PHPE9568F34-D428-11d2-A769-00AA001ACF42)|\\x[0-9a-zA-Z]{2})
      '');
    }
    // (with lib.attrsets;
      mapAttrs' (
        name: value:
          nameValuePair
          "fail2ban/filter.d/${name}.local"
          {text = lib.mkDefault (lib.mkAfter value.filter);}
      )
      jails);
}
