{
  config,
  pkgs,
  ...
}: let
  host = "127.0.0.1";
  port = 6969;
  anubis-port = "8923";
in {
  imports = [
    ./nginx.nix
    ./anubis.nix
    ./gitea-actions-runner.nix
  ];
  services.forgejo = {
    enable = true;

    lfs.enable = true;
    database.type = "postgres";

    settings = {
      DEFAULT = {
        APP_NAME = "An idiot admires complexity. A genius admires simplicity";
      };

      repository = {
        ENABLE_PUSH_CREATE_USER = true;
      };

      server = {
        DOMAIN = "git.spoodythe.one";
        HTTP_PORT = port;
        ROOT_URL = "https://git.spoodythe.one";
      };

      security.REVERSE_PROXY_TRUSTED_PROXIES = "127.0.0.0/24";

      service.DISABLE_REGISTRATION = true;
      actions = {
        ENABLED = true;
        DEFAULT_ACTIONS_URL = "https://code.forgejo.org";
      };
      federation.ENABLED = false;
    };
  };

  # NGINX
  services.nginx.virtualHosts."git.spoodythe.one" = {
    forceSSL = true;
    enableACME = true;
    locations."/" = {
      extraConfig = ''
        proxy_set_header Host $host;
        proxy_set_header X-Real-Ip $remote_addr;
        client_max_body_size 100M;
        # using nix `proxyPass = ...` doesnt work since it needs to be after the headers for anubis to work
        proxy_pass http://${host}:${toString anubis-port};
      '';
    };
  };

  services.anubis.instances.forgejo = {
    settings = {
      BIND = ":${toString anubis-port}";
      TARGET = "http://${host}:${toString port}";
      METRICS_BIND = ":${toString 32000}";

      POLICY_FNAME = pkgs.writeText "challenge_all.json" (builtins.toJSON {
        bots = [
          {
            name = "well-known";
            path_regex = "^/.well-known/.*$";
            action = "ALLOW";
          }
          {
            name = "favicon";
            path_regex = "^/favicon.ico$";
            action = "ALLOW";
          }
          {
            name = "robots-txt";
            path_regex = "^/robots.txt$";
            action = "ALLOW";
          }
          {
            name = "allow-summary";
            path_regex = "^/.*/.*/-/summary-card$";
            action = "ALLOW";
          }
          {
            name = "git";
            user_agent_regex = "git";
            action = "ALLOW";
          }
          {
            name = "generic-browser";
            user_agent_regex = ".*";
            action = "CHALLENGE";
          }
        ];
        dnsbl = false;
        status_codes.CHALLENGE = 200;
        status_codes.DENY = 200;
      });

      OG_PASSTHROUGH = true;
      OG_EXPIRY_TIME = "1h";
    };
  };
}
